#!/usr/bin/env jpython

#
# output format (fields tab-separated):
#
# desired G name, time since start (s), correct (0/1), actual G index,
# actual G name, rotation, horizontally flipped, vertically flipped,
# backwards
#
# Fields after correct do not appear if it's correct.  Fields after
# actual G name do not appear if actual G isn't a real one (e.g., if
# it's DOT).

import os, sys, re, glob, string, os.path, time, traceback

usage = sys.argv[0] + ": usage: " + sys.argv[0] + " stimuli.gsa participantPath\n"

GS_FILE_REGEX = re.compile("(?P<prefix>.*/)?(?P<pid>\d+)-(?P<phase>\d)_(?P<month>\d+)-(?P<day>\d+).gsa")
def gsFileRename(name):
	match = GS_FILE_REGEX.match(name)
	return "%s%s%s%s" % match.group('pid', 'month', 'day', 'phase')

def gsFileCompare(a, b):
	aString = gsFileRename(a)
	bString = gsFileRename(b)
	if aString == bString:
		return 0
	elif aString > bString:
		return 1
	else:
		return -1

def gsFileSort(gsFiles):
	newFiles = gsFiles
	if len(gsFiles) > 1:
		newFiles.sort(gsFileCompare)
		match = GS_FILE_REGEX.match(newFiles[0])
		origDate = match.group('month', 'day')
		
		for i in range(len(newFiles))[1:]:
			match = GS_FILE_REGEX.match(newFiles[i])
			date = match.group('month', 'day')
			if origDate != date:
				part2 = newFiles[i:]
				part2.reverse()
				newFiles[i:] = part2
				return newFiles
	return newFiles

numArgs = len(sys.argv)
if numArgs == 3:
	# name specified
	stimulusFile = sys.argv[1]
	textFile = sys.argv[2] + ".txt"
	participantID = string.atoi(os.path.basename(sys.argv[2]))
	gsFiles = glob.glob("%s-*.gsa" % sys.argv[2])
	gsFileSort(gsFiles)
elif numArgs >= 5 and sys.argv[1] == "-old":
 	stimulusFile = sys.argv[2]
 	textFile = sys.argv[3]
 	# training testing retesting retraining.  Only some may actually
 	# appear.
 	gsFiles = sys.argv[4:]
else:
	sys.stderr.write(sys.argv[0] + ": error: wrong # of arguments\n")
	sys.stderr.write(usage)
	sys.exit(-1)

from edu.berkeley.guir.gesture import GestureSetFrame, GestureSet, GestureSetDisplay, GestureDisplay
from java.io import *
from java.lang import *
from java.awt import *
from javax.swing import *

class MatchPanel(JPanel):
	"panel for a single gesture"

	def __init__(self, drawing, gestureSet):
		"""Drawing is the Gesture the participant drew.
		gestureSet is the set being learned."""
		JPanel.__init__(self, BorderLayout())
		self.gestureSet = gestureSet
		# first do the part that chooses the gesture
		gesturePanel = JPanel()
		drawing.normalize()
		self.drawnGesture = drawing

		drawnDisplay = GestureDisplay(self.drawnGesture)
		drawnDisplay.setOffset(5, 5);
		drawnDisplay.setBorder(BorderFactory.createLoweredBevelBorder())
		gesturePanel.add(drawnDisplay)
		self.selectedGestureDisplay = GestureDisplay()
		self.selectedGestureDisplay.setOffset(5, 5);
		self.selectedGestureDisplay.setBorder(BorderFactory.createLoweredBevelBorder())
		gesturePanel.add(self.selectedGestureDisplay)
		self.add(gesturePanel, BorderLayout.WEST)
		choicePanel = Box.createVerticalBox()
		self.categorySelected = ''

		# now do the part that chooses other stuff
		rotationPanel = JPanel()
		rotationPanel.add(JLabel('Rotation:'))
		self.rotationField = JTextField('0')
		self.rotationField.setColumns(5);
		rotationPanel.add(self.rotationField)
		choicePanel.add(rotationPanel)
		self.horizontalFlippedButton = JCheckBox('Horizontal flip')
		choicePanel.add(self.horizontalFlippedButton)
		self.verticalFlippedButton = JCheckBox('Vertical flip')
		choicePanel.add(self.verticalFlippedButton)
		self.backwardsButton = JCheckBox('Backwards')
		choicePanel.add(self.backwardsButton)
		choicePanel.add(Box.createVerticalGlue())
		self.add(choicePanel, BorderLayout.EAST)

		radioPanel = JPanel()
		radioPanel.setLayout(BoxLayout(radioPanel,BoxLayout.Y_AXIS))
		radioPanel.setBorder(BorderFactory.createEtchedBorder())
		self.dotButton = JRadioButton('Is a dot', actionPerformed = lambda e, gd=self.selectedGestureDisplay: gd.setGesture(None))
		radioPanel.add(self.dotButton)

		self.blankButton = JRadioButton('Is blank', actionPerformed = lambda e, gd=self.selectedGestureDisplay: gd.setGesture(None))
		radioPanel.add(self.blankButton)

		self.realButton = JRadioButton('dummy')

		self.buttonGroup = ButtonGroup()
		self.buttonGroup.add(self.dotButton)
		self.buttonGroup.add(self.blankButton)
		self.buttonGroup.add(self.realButton)
		if drawing.getPoints().npoints == 0:
			self.blankButton.getModel().setSelected(1)

		choicePanel.add(radioPanel)

		self.setBorder(BorderFactory.createEtchedBorder())

	def setCategory(self, gestureCategory):
		self.categorySelected = gestureCategory
		self.selectedGestureDisplay.setGesture(gestureCategory.gestureAt(0))
		self.buttonGroup.setSelected(self.realButton.getModel(), 1)
	def getCategory(self):
		return self.categorySelected
	def getCatIndex(self):
		return self.gestureSet.indexOf(self.categorySelected)
	def getRotation(self):
		return int(self.rotationField.getText())
	def isHorizFlipped(self):
		return self.horizontalFlippedButton.isSelected()
	def isVertFlipped(self):
		return self.verticalFlippedButton.isSelected()
	def isBackwards(self):
		return self.backwardsButton.isSelected()
	def isDot(self):
		return self.dotButton.isSelected()
	def isBlank(self):
		return self.blankButton.isSelected()
	def isReal(self):
		return self.realButton.isSelected()
				
currentMatchFrame = []

# Frame holding a list of things to match
class MatchFrame(JFrame):
	# responses is a list of strings from the .txt file
	def __init__(self, stimuliSet, inputSet, responses, outFileName):
		JFrame.__init__(self, 'Match gestures')
		self.responses = responses
		self.matchesPanel = MatchesPanel(stimuliSet, inputSet, responses)
		self.outFileName = outFileName
		scroller = JScrollPane(self.matchesPanel)
		scroller.setPreferredSize(Dimension(500,600))
		contents = self.getContentPane()
		contents.setLayout(BorderLayout())
		contents.add(scroller, BorderLayout.CENTER)
		controlPanel = JPanel()
		saveButton = JButton('Save', actionPerformed=lambda e, f=self: f.saveResults())
		controlPanel.add(saveButton)
		closeButton = JButton('Close', actionPerformed=lambda e, f=self: f.setVisible(0))
		controlPanel.add(closeButton)
		self.componentResized.append(lambda e, c=self.getRootPane(): c.setPreferredSize(c.getSize()))

		contents.add(controlPanel, BorderLayout.SOUTH)

	def gestureCount(self):
		return self.matchesPanel.gestureCount()

	def saveResults(self):
		print "Writing to file '%s'" % self.outFileName
		try:
			outFile = open(self.outFileName, 'w')
			self.matchesPanel.saveResults(outFile)
			outFile.close()
		except:
			traceback.print_exc(file=sys.stdout)
			JOptionPane.showMessageDialog(None, 'Error writing results.  See console for more information.', 
						      'Save Error',
						      JOptionPane.ERROR_MESSAGE)

class Response:
	"Members: year, month, day, hour, minute, second, name, correct"
	regex = re.compile("(\d+)/(\d+)>(\d+):(\d+):(\d+)\t(.+)\t(.*CORRECT)")
	def __init__(self, desiredName, responseStr):
		map = Response.regex.search(responseStr)
		if map:
			self.desiredName = desiredName
			self.year = 1999
			(self.month, self.day, self.hour, self.minute, self.second,
			 self.name, self.correct) = map.groups()
			# hack to convert to military time
			if self.hour < 8:
				self.hour = self.hour + 12
			self.secondsSinceEpoch = time.mktime((self.year,
							      self.month,
							      self.day,
							      self.hour,
							      self.minute,
							      self.second,
							      -1, -1, -1))
			self.valid = 1
		else:
			self.valid = 0
		self.widget = None
	def setWidget(self, widget):
		self.widget = widget
	def toString(self, startTime):
		"startTime should be in seconds since the start of the epoch"
		result = "%s\t%d" % (self.desiredName, self.secondsSinceEpoch - startTime)
		if self.widget:
			result = result + '\t0'
			if self.widget.isReal():
				result = result + \
					 "\t%d\t%s\t%d\t%d\t%d\t%d" % \
					 (self.widget.getCatIndex(),
					  self.widget.getCategory().getName(),
					  self.widget.getRotation(),
					  self.widget.isHorizFlipped(),
					  self.widget.isVertFlipped(),
					  self.widget.isBackwards())
			elif self.widget.isBlank():
				result = result + "\t-1\tBLANK"
			elif self.widget.isDot():
				result = result + "\t-2\tDOT"
			else:
				result = result + "\t-3\tBOGUS"
		else:
			result = result + '\t1'
		return result
	def getHeader(self):
		return "Desired Name\tSecondSinceStart\tIs Correct\tDrawn Index\tDrawn Name\tRotation\tIs Horiz Flipped\tIs Vert Flipped\tIs Backwards"

class MatchesPanel(Box):
	# responses is a list of strings from the .txt file
	def __init__(self, stimuliSet, inputSet, responseStrings):
		Box.__init__(self, BoxLayout.Y_AXIS)
		i = 0
		numGestures = inputSet.size()
		print "MatchesPanel: # gestures = %d\t# responses = %d" % \
		      (numGestures, len(responseStrings))
		if numGestures != len(responseStrings):
			JOptionPane.showMessageDialog(None, 'Warning: Mismatch between log and gesture files.', 
						      'Data Error',
						      JOptionPane.ERROR_MESSAGE)
		def selector(e, container=self):
			global currentMatchFrame
			newSelection = e.getComponent()
			oldSelection = container.selectedPanel
			if (oldSelection):
				oldSelection.setBorder(BorderFactory.createEtchedBorder())
				if (newSelection != oldSelection):
					newSelection.setBorder(BorderFactory.createLoweredBevelBorder())
					container.selectedPanel = newSelection
					currentMatchFrame = container
				else:
					container.selectedPanel = ''
			else:
				newSelection.setBorder(BorderFactory.createLoweredBevelBorder())
				container.selectedPanel = newSelection
				currentMatchFrame = container
		self.responses = []
		for responseStr in responseStrings:
			category = inputSet.categoryAt(i)
			desiredName = category.getName()
			response = Response(desiredName, responseStr)
			if response.valid:
				if (response.correct == "INCORRECT"):
					g = category.gestureAt(0)
					p = MatchPanel(g, stimuliSet)
					p.mouseClicked.append(selector)
					self.add(p)
					response.setWidget(p)
				self.responses.append(response)
			i = i + 1
			if (i >= numGestures):
				break
		self.selectedPanel = ''

	def gestureCount(self):
		return self.getComponentCount()

	def saveResults(self, outFile):
		startTime = self.responses[0].secondsSinceEpoch
		for response in self.responses:
			outFile.write(response.toString(startTime) + '\n')

# global functions

def loadGestureSet(filename):
	gsFrame = GestureSetFrame('loading', 0)
	gsFrame.openFile(File(filename))
	return gsFrame.getGestureSetDisplay().getGestureSet()

def splitFile(name):
	lines = open(name).readlines()
	sections = splitText(lines)
	return sections

def splitText(lines):
	sections = []
	section = []
	for line in lines:
		if (re.search(">", line)):
			section.append(line)
		else:
			if (section != []):
				sections.append(section)
				section = []
	if (section != []): sections.append(section)
	return sections

def normalizeGestureSet(gs):
	for gcIndex in range(gs.size()):
		category = gs.categoryAt(gcIndex)
		for gIndex in range(category.size()):
			category.gestureAt(gIndex).normalize()

		
#
# main
#

# UI setup
UIManager.put("TitledBorder.titleColor", Color.black)
UIManager.put("Label.foreground", Color.black)

# real main
print "Parsing text file..."
textBlocks = splitFile(textFile)

print "Loading stimuli..."
stimulusSet = loadGestureSet(stimulusFile)
print "Loaded"
normalizeGestureSet(stimulusSet)
stimulusFrame = JFrame('Stimuli: ' + stimulusFile)
contents = stimulusFrame.getContentPane()
contents.setLayout(BorderLayout())

gsDisplay = GestureSetDisplay(stimulusSet, 0)
scroller = JScrollPane(gsDisplay)
scroller.setMinimumSize(Dimension(0, 0))
scroller.setPreferredSize(Dimension(895,1075))
gsDisplay.setScroller(scroller)

contents.add(scroller, BorderLayout.CENTER)

def selectionHandler(e):
	if ((e.getStateChange() == e.SELECTED) and (currentMatchFrame != [])):
		p = currentMatchFrame.selectedPanel
		p.setCategory(e.getItem().getGestureCategory())
		SwingUtilities.getRoot(currentMatchFrame).pack()
		e.getItem().setSelected(0)

gsDisplay.itemStateChanged.append(selectionHandler)

masterFrame = JFrame('Selector')
buttonPanel = masterFrame.getContentPane()
buttonPanel.setLayout(BoxLayout(buttonPanel, BoxLayout.Y_AXIS))

def outputFileName(inputName):
	GS_FILE_REGEX = re.compile("(?P<prefix>.*/)?(?P<pid>\d+)-(?P<trial>\d)_(?P<month>\d+)-(?P<day>\d+)(?P<incomplete>-Incomplete)?.gsa")
	match = GS_FILE_REGEX.match(inputName)
	prefix = match.group('prefix')
	if prefix == None:
		prefix = ""
	(pid, trial, month, day) = map(string.atoi, match.group('pid', 'trial', 'month', 'day'))
	return "%s%03d-%d_%02d-%02d-cm.txt" % (prefix, pid, trial, month, day)

print "Loading drawn gestures..."
matchFrames = []
for i in range(min(len(gsFiles),len(textBlocks))):
	print "\tPhase %d..." % i
	text = textBlocks[i]
	inputSet = loadGestureSet(gsFiles[i])
	normalizeGestureSet(inputSet)
	matchFrames.append(MatchFrame(stimulusSet, inputSet, text, outputFileName(gsFiles[i])))
	matchFrames[i].setTitle(gsFiles[i])
	matchFrames[i].pack()
	button = JButton(gsFiles[i])
	button.actionPerformed = lambda e, f=matchFrames[i]: f.show()
	if (matchFrames[i].gestureCount() == 0):
		button.setToolTipText('No errors')
		button.setForeground(Color.blue)
	buttonPanel.add(button)

print "Ready to start"

mb = JMenuBar()
fileMenu = JMenu('File')
item = JMenuItem('Quit')
item.actionPerformed = lambda e: sys.exit(0)
fileMenu.add(item)

mb.add(fileMenu)
masterFrame.getRootPane().setJMenuBar(mb)

stimulusFrame.pack()
stimulusFrame.show()

masterFrame.pack()
masterFrame.show()
